// Apple ][ Cassette Audio Decoder
// Copyright (c) Sebastian Kienzl <seb at riot dot org>
// 
// $Id: main.cpp 20 2008-09-15 19:13:36Z seb $
//
// Information on the Waveform taken from "Woz Wonderbook",
// ADTPro-Source and Monitor-ROM disassembly
//
// Should also compile under Linux and other POSIX-platforms
//
// Input file must be a 8-bit unsigned data

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef signed char s8;

class ByteBuffer {
public:
	ByteBuffer() : data(0), allocLen(0), len(0) {}
	~ByteBuffer() { if( data ) free( data ); }

	void addByte( s8 d ) {
		if( len >= allocLen ) {
			#define ALLOC_STEP 1024
			data = (s8*)realloc( data, allocLen + ALLOC_STEP );
			memset( data + allocLen, 0, ALLOC_STEP );
			allocLen += ALLOC_STEP;
		}
		data[ len++ ] = d;
	}

	void removeByte() {
		if( len )
			--len;
	}

	int getLength() { return len; }
	const s8* getData() { return data; }
private:
	s8* data;
	int allocLen;
	int len;
};

ByteBuffer decodedData;

#define HEADER_SKIP 0x80

enum e_DecodeState {
	INIT,
	WAITING_SYNC,
	WAITING_HEADER,
	DATA
} decodeState = INIT;

int notifyNoZeroCrossingAfter = 0;

int syncCount = 0;
int syncLastPeriod = 0;
float syncSampleFreq = 0;
int headerCount;

int dataDecidePeriod;
bool data2ndHalf;
int dataBitCount;
int data;
int dataCheckSum;
int dataByteCount;
int dataLast;

const char* getTimeString( int index ) {
	static char buffer[ 64 ];
	sprintf( buffer, "%.2f secs", index/syncSampleFreq );
	return buffer;
}

void printTime( int index ) {
	printf( "%s\t: ", getTimeString(index) );
}

void decodeError( int index ) {
	printTime( index );
	printf( "decoding error\n" );
	exit( 1 );
}


void decodedByte( int byte ) {
	decodedData.addByte( byte );
	dataLast = byte;
	dataByteCount++;
}

void zeroCrossing( int index, int diff ) {
	switch( decodeState ) {
	case INIT:
		decodeState = WAITING_SYNC;
		syncCount = 0;
		syncLastPeriod = 0;
		headerCount = 0;
		data2ndHalf = false;
		dataBitCount = 0;
		data = 0;
		dataCheckSum = 0xff;
		dataByteCount = 0;

		// fallthrough on purpose
	case WAITING_SYNC:
		if( abs( syncLastPeriod - diff ) < diff/10 ) {
			syncCount++;
			if( syncCount > 100 ) {
				syncSampleFreq = 2*diff/.0013f;
				dataDecidePeriod = (int)( .5 + syncSampleFreq * 0.000375 );
				printTime(index);
				printf( "sync found, period is %d, samplefreq is ~%.2f\n", diff, syncSampleFreq );
				decodeState = WAITING_HEADER;
				return;
			}
		}
		else {
			syncCount = 0;
		}
		syncLastPeriod = diff;

		break;

	case WAITING_HEADER:
		if( diff < syncLastPeriod*2/3 ) {
			if( !headerCount ) {
				printTime(index);
				printf( "data starts\n" );
			}
			if( ++headerCount >= 2 ) {
				decodeState = DATA;
				return;
			}
		}
		else if( headerCount )
			decodeError(index);
		break;

	case DATA:
		notifyNoZeroCrossingAfter = 5*dataDecidePeriod;
		if( data2ndHalf ) {
			if( diff > 2*dataDecidePeriod )
				decodeError(index);
			data2ndHalf = false;
			return;
		}

		data <<= 1;
		if( diff > dataDecidePeriod ) {
			data |= 1;
		}
		if( ++dataBitCount >= 8 ) {
			decodedByte( data );
			dataCheckSum ^= data;
			data = 0;
			dataBitCount = 0;
		}

		data2ndHalf = true;

		break;
	default:
		break;

	}
}

void noZeroCrossing( int index, int diff ) {
	dataCheckSum ^= dataLast;
	dataByteCount--;
	decodedData.removeByte();

	printTime( index );
	printf( "data ends: %d bytes, checksum %s\n\n", dataByteCount, dataCheckSum == dataLast ? "OK" : "FAIL" );

	decodeState = INIT;
	notifyNoZeroCrossingAfter = 0;
}


int main( int argc, char** argv )
{
	printf( "Apple ][ Cassette Audio Decoder\n" \
		"(c) 2008 seb at riot dot org\n" \
		"$Rev: 20 $\n\n" );

	if( argc < 3 ) {
		fprintf( stderr, "%s <infile> <outfile>\nInput file must be 8-bit unsigned data (WAV, but can be RAW)\n\n", argv[0] );
		return 1;
	}

	FILE* fin = fopen( argv[1], "rb" );
	if( !fin ) {
		fprintf( stderr, "can't open file for reading: %s\n", argv[1] );
		return 1;
	}

	fseek( fin, 0, SEEK_END );
	int dataSize = ftell( fin );
	dataSize -= HEADER_SKIP;
	fseek( fin, HEADER_SKIP, SEEK_SET );
	
	s8* data = new s8[ dataSize ];
	fread( data, dataSize, 1, fin );
	fclose( fin );

	int sum = 0;
	for( int i = 0; i < dataSize; i++ ) {
		data[i] += 0x80;	// unsigned -> signed
		sum += data[i];
	}
	float DC = (float)( sum / dataSize );
	printf( "DC=%.2f\n\n", DC );

	int lastZeroCrossing = -1;
	
	int signCount = 0;
	int sign = 1;
	int oldSign = 0;

	for( int i = 0; i < dataSize; i++ ) {
		float val = ( float )data[ i ];
		val -= DC;

		if( val >= 0 )
			sign = 1;
		else
			sign = 0;
		
		if( oldSign == sign ) {
			signCount++;
		}
		else
			signCount = 0;
		oldSign = sign;

		if( signCount == 5 ) {
			if( lastZeroCrossing >= 0 ) {
				int diff = i - lastZeroCrossing;
				zeroCrossing( i, diff );
			}
			lastZeroCrossing = i;
		}

		if( notifyNoZeroCrossingAfter > 0 && notifyNoZeroCrossingAfter < ( i - lastZeroCrossing ) ) {
			noZeroCrossing( i, i - lastZeroCrossing );
		}

	}

	printf( "Total: %d bytes\n", decodedData.getLength() );

	FILE* fout = fopen( argv[2], "wb" );
	if( !fout ) {
		fprintf( stderr, "can't open file for writing: %s\n", argv[2] );
		return 1;
	}

	fwrite( decodedData.getData(), decodedData.getLength(), 1, fout );
	fclose( fout );
}
